// ========================================
// GRAPHIQUE D'ÉVOLUTION D3.js
// ========================================

let evolutionChart = null;

function initEvolutionChart() {
    const container = document.getElementById('evolutionChartContainer');
    if (!container) return;
    
    // Nettoyer le conteneur
    container.innerHTML = '';
    
    // Dimensions
    const width = container.clientWidth;
    const height = container.clientHeight;
    const margin = { top: 40, right: 50, bottom: 40, left: 50 };
    const chartWidth = width - margin.left - margin.right;
    const chartHeight = height - margin.top - margin.bottom;
    
    // Créer le SVG
    const svg = d3.select(container)
        .append('svg')
        .attr('width', width)
        .attr('height', height)
        .attr('class', 'tv-d3-chart');
    
    const g = svg.append('g')
        .attr('transform', `translate(${margin.left}, ${margin.top})`);
    
    // Récupérer les données
    const data = prepareEvolutionData();
    if (!data || data.length === 0) {
        console.warn('Aucune donnée pour le graphique d\'évolution');
        return;
    }
    
    // Échelles
    const xScale = d3.scaleLinear()
        .domain(d3.extent(data[0].values, d => d.year))
        .range([0, chartWidth]);
    
    const yScale = d3.scaleLinear()
        .domain([0, 100]) // Pourcentage de 0 à 100%
        .range([chartHeight, 0]);
    
    // Axes
    const xAxis = d3.axisBottom(xScale)
        .tickFormat(d3.format('d'))
        .ticks(8);
    
    const yAxis = d3.axisLeft(yScale)
        .tickFormat(d => d + '%')
        .ticks(5);
    
    // Grille
    g.append('g')
        .attr('class', 'tv-d3-grid')
        .call(d3.axisLeft(yScale)
            .tickSize(-chartWidth)
            .tickFormat('')
        );
    
    // Axes
    g.append('g')
        .attr('class', 'tv-d3-axis')
        .attr('transform', `translate(0, ${chartHeight})`)
        .call(xAxis);
    
    g.append('g')
        .attr('class', 'tv-d3-axis')
        .call(yAxis);
    
    // Ligne de génération
    const line = d3.line()
        .x(d => xScale(d.year))
        .y(d => yScale(d.rate))
        .curve(d3.curveMonotoneX);
    
    // Couleurs des chaînes
    const colorScale = d3.scaleOrdinal()
        .domain(data.map(d => d.channel))
        .range(d3.schemeSet2);
    
    // Ajouter les lignes
    const lines = g.selectAll('.tv-d3-line')
        .data(data)
        .enter()
        .append('path')
        .attr('class', 'tv-d3-line')
        .attr('d', d => line(d.values))
        .attr('fill', 'none')
        .attr('stroke', d => colorScale(d.channel))
        .attr('stroke-width', 3)
        .attr('stroke-linejoin', 'round')
        .attr('stroke-linecap', 'round');
    
    // Ajouter les points
    const points = g.selectAll('.tv-d3-point')
        .data(data.flatMap(d => d.values.map(v => ({ ...v, channel: d.channel }))))
        .enter()
        .append('circle')
        .attr('class', 'tv-d3-point')
        .attr('cx', d => xScale(d.year))
        .attr('cy', d => yScale(d.rate))
        .attr('r', 3)
        .attr('fill', d => colorScale(d.channel))
        .attr('stroke', '#fff')
        .attr('stroke-width', 1)
        .on('mouseover', handlePointMouseOver)
        .on('mouseout', handlePointMouseOut);
    
    // Légende
    const legend = svg.append('g')
        .attr('class', 'tv-d3-legend')
        .attr('transform', `translate(${width - 200}, ${margin.top})`);
    
    const legendItems = legend.selectAll('.tv-d3-legend-item')
        .data(data)
        .enter()
        .append('g')
        .attr('class', 'tv-d3-legend-item')
        .attr('transform', (d, i) => `translate(0, ${i * 25})`)
        .on('click', handleLegendClick);
    
    legendItems.append('rect')
        .attr('width', 15)
        .attr('height', 15)
        .attr('fill', d => colorScale(d.channel))
        .attr('stroke', '#fff')
        .attr('stroke-width', 1)
        .attr('rx', 2);
    
    legendItems.append('text')
        .attr('x', 20)
        .attr('y', 12)
        .text(d => d.channel)
        .attr('fill', '#fff')
        .attr('font-size', '12px')
        .attr('font-family', 'Segoe UI, sans-serif');
    
    // Tooltip
    const tooltip = d3.select('body')
        .append('div')
        .attr('class', 'tv-d3-tooltip')
        .style('opacity', 0)
        .style('position', 'absolute')
        .style('background', 'rgba(40, 40, 40, 0.95)')
        .style('border', '1px solid rgba(0, 168, 255, 0.5)')
        .style('border-radius', '8px')
        .style('padding', '12px')
        .style('color', '#fff')
        .style('font-family', 'Segoe UI, sans-serif')
        .style('font-size', '12px')
        .style('pointer-events', 'none')
        .style('z-index', '1000')
        .style('box-shadow', '0 5px 15px rgba(0, 0, 0, 0.5)');
    
    function handlePointMouseOver(event, d) {
        tooltip.transition()
            .duration(200)
            .style('opacity', 0.95);
        
        tooltip.html(`
            <div style="margin-bottom: 5px; font-weight: bold; color: ${colorScale(d.channel)}">
                ${d.channel}
            </div>
            <div>Année: <strong>${d.year}</strong></div>
            <div>Taux femmes: <strong>${d.rate}%</strong></div>
            <div>Taux hommes: <strong>${100 - d.rate}%</strong></div>
        `)
            .style('left', (event.pageX + 15) + 'px')
            .style('top', (event.pageY - 15) + 'px');
        
        d3.select(event.target)
            .transition()
            .duration(200)
            .attr('r', 6)
            .attr('stroke-width', 2);
    }
    
    function handlePointMouseOut(event, d) {
        tooltip.transition()
            .duration(200)
            .style('opacity', 0);
        
        d3.select(event.target)
            .transition()
            .duration(200)
            .attr('r', 3)
            .attr('stroke-width', 1);
    }
    
    function handleLegendClick(event, d) {
        const lineElement = d3.select(event.currentTarget);
        const isActive = lineElement.classed('inactive');
        
        // Toggle l'état
        lineElement.classed('inactive', !isActive);
        
        // Toggle la visibilité de la ligne
        const lineIndex = data.indexOf(d);
        const linePath = d3.selectAll('.tv-d3-line').filter((data, index) => index === lineIndex);
        const linePoints = d3.selectAll('.tv-d3-point').filter(point => point.channel === d.channel);
        
        if (isActive) {
            linePath.transition().duration(300).style('opacity', 1);
            linePoints.transition().duration(300).style('opacity', 1);
        } else {
            linePath.transition().duration(300).style('opacity', 0.3);
            linePoints.transition().duration(300).style('opacity', 0.3);
        }
    }
    
    evolutionChart = { svg, g, xScale, yScale, colorScale };
}

function prepareEvolutionData() {
    if (!window.CSV_DATA?.years?.tv) return [];
    
    const data = [];
    const years = Object.keys(window.CSV_DATA.years.tv).sort();
    
    // Sélectionner les chaînes principales
    const mainChannels = ['TF1', 'France 2', 'France 3', 'ARTE', 'M6', 'Canal+'];
    
    mainChannels.forEach(channel => {
        const channelData = {
            channel: channel,
            values: []
        };
        
        years.forEach(year => {
            const yearData = window.CSV_DATA.years.tv[year];
            if (yearData && yearData[channel]) {
                channelData.values.push({
                    year: parseInt(year),
                    rate: parseFloat(yearData[channel])
                });
            }
        });
        
        if (channelData.values.length > 0) {
            data.push(channelData);
        }
    });
    
    return data;
}

// Redimensionner le graphique
function resizeEvolutionChart() {
    if (evolutionChart) {
        initEvolutionChart();
    }
}

// Initialiser au chargement
document.addEventListener('DOMContentLoaded', function() {
    setTimeout(() => {
        initEvolutionChart();
        
        // Redimensionner lors du resize de la fenêtre
        window.addEventListener('resize', resizeEvolutionChart);
    }, 500);
});

// Export
window.initEvolutionChart = initEvolutionChart;
window.resizeEvolutionChart = resizeEvolutionChart;